/*!************************************************************************
 * FILE :         clAudioSourceController.h
 * SW-COMPONENT:  Audio Engine
 * DESCRIPTION:
 *
 * The main task of AudioSourceController is to perform source-switches
 * by activating and deactivating AudioSources.
 * It handles Priority checks by applying Stack-Rules, defined in clStackRules.cpp
 * AudioSources are kept in a stack, where the priority queue is realised.
 *
 * The typical straight forward approach for a source-switch is:
 *
 * - Client requests an AudioSource to be played
 * - Controller adds the requested AudioSource to the stack, applying
 *   priority rules etc.
 * - If the new AudioSource is on top of the stack, it has to be started
 * - Before, the previous active source has to be stopped.
 * - So the sequence is:
 *   - Start to mute the current active source and wait until the mute-ramp
 *   for this source has finished.
 *   - Then start the new source
 *
 * Things become a littel bit more complicated, when mix sources have to
 * be added, because the means, that more than on source can be active
 * at the same time.
 *
 * So the advanced approach is:
 * - Client requests an AudioSource to be played
 * - AudioSourceController stopps all sources, which should not be active
 * after applying the StackRules.
 * - Wait until all stopping source have finished the mute-ramp.
 * - AudioSourceController starts all sources, which should be active after
 * applying the StackRules.
 *
 *
 *
 * AUTHOR:        CM-DI/PJ-VW34 Steinle
 * COPYRIGHT:     (c) 2006 Blaupunkt Werke
 *************************************************************************/
#ifndef _AUDIOSOURCECONTROLLER_H_
#define _AUDIOSOURCECONTROLLER_H_

// needed includes
#include "AudioStack/clStack.h"
#include "AudioStack/clDiagStack.h"
#include "AudioStack/AudioSources/clAudioSource.h"
#include "AudioStack/InterfaceAudioStack.h"

#include "AudioStack/AudioSources/clFactory_AudioSourceClass.h"

#ifndef _DO_NOT_USE_BOSCH_OSAL_
//#define SYSTEM_S_IMPORT_INTERFACE_LIST
//#define SYSTEM_S_IMPORT_INTERFACE_ALGORITHM
//#include <stl_pif.h>
#else
#include <list>
#endif

// use this bitmask in u32Param2 wenn sending an OFF message and the play cmd counter should be ignored
#define BITMASK_FORCE_OFF 0x00FF0000

namespace AudioStack
{
namespace AudioSource
{

class clIAudioSourceControllerObserver;

/**
 * \brief
 * The main task of AudioSourceController is to perform source-switches
 * by activating and deactivating AudioSources.
 * It handles Priority checks by applying Stack-Rules, defined in clStackRules.cpp
 * AudioSources are kept in a stack, where the priority queue is realised.
 */
class clAudioSourceController : public clAudioSourceObserver
{
public:

   tVoid vInvalidate();
   /** Reset Stack */
   tVoid vResetStack();
   tVoid vResetStack(int sinkID);
   /** guides the command to the stack object itself to trace the current stack situation */
   tVoid vTraceStack();

   tVoid vTraceBackupStack() const;

   /** stores the pointer to the observer in a member, used to inform about audio source switch processes */
   tVoid vSetObserver(clIAudioSourceControllerObserver* poObserver){m_poObserver = poObserver;};
   /** start the given source, used for foreground sources */
   tBool bON(SourceID srcID, tU32 userData=0);
   tBool bON(SourceID srcID, tU16 sinkID, tU32 userData);
   /** stopps the given source, used for background and foreground sources */
   tBool bOFF(SourceID srcID, tU32 userData=0);
   tBool bOFF(SourceID srcID, tU16 sinkID, tU32 u32UserData);
   /** start the given source, used for background sources */
   tBool bSELECT(SourceID srcID, tU32 userData=0);
   tBool bSELECT(SourceID srcID, tU16 sinkID, tU32 u32UserData);
   /** start the given source, used for mix sources */
   tBool bMIX(SourceID srcID, tU32 userData=0);
   tBool bMIX(SourceID srcID, tU16 sinkID, tU32 userData);
   /** stopps the given source, used for mix sources */
   tBool bMIXOFF(SourceID srcID, tU32 userData=0);
   tBool bMIXOFF(SourceID srcID, tU16 sinkID, tU32 userData=0);
   /** returns a boolean, indicating if a mute source is on stack */
//   tBool bIsAudioMuted();
   /** returns a boolean, indicating if the current entertainment source is paused */
//   tBool bIsEntertainmentSourcePaused();
   /** remove exclusive source in ramp up/down transition*/
   tVoid vHandleSrcActError(SourceID errSrc);
   /** check if a source is on stack */
   static tBool bIsSourceOnStack(SourceID srcID);
   static tBool bIsSourceOnStack(SourceID srcID, tU16 SinkID);

    static tBool bIsSourceOnBackUpStack(SourceID srcID);
  // ToDo: check if we need it also with sink parameter

   /** check if a one more non-stackable source is on stack */
   static tBool bIsAnotherEntSrcOnStack(SourceID srcID);

   /** returns a boolean, indicating if the source is current active source */
   tBool bIsCurrentActiveAudiosource(SourceID srcID);


   /** check if source is allowed to be on stack, request will be routed to stack object */
   clSourceClass::allow_t enCheckSource(clAudioSource* poAudioSource);


//   SourceID u8GetTopOfStackAudiosource();

   static clAudioSource* pcoGetTopOfStack();
   static clAudioSource* pcoGetTopOfStackMixsource();
   static clAudioSource* pcoGetSecondMixSource();

   static tVoid vGetStackList(std::list<  clAudioSource * > &list);

   //daw2hi added 21.07.2016
   static tBool GetSinkIDsForSource(SourceID srcID, std::vector<int> &sinkIDs);

   clAudioSource* pcoGetCurrentActiveAudiosource();
   clAudioSource* pcoGetPossibleNextSource();
   clAudioSource* pcoGetFirstNonMuteSrcFromStack();

   /**
    * return singleton instance
    */
   static clAudioSourceController& getInstance();

   // daw2hi: Remote Ctrl commands
   static am_Error_e vRemoteCtrlStart();
   static am_Error_e vRemoteCtrlEnd();

   tBool bIsTopOfBackUpStackMuteAndOn();

private:


  typedef enum
   {
    topOfStack = 1,
    underlayingInStack = 2
  } switchingType_t;

  typedef enum
   {
    STATE_NEUTRAL  = 0,
    STATE_MUTING   = 1,
    STATE_DEMUTING = 2
  } tenStates;

  //private due to singleton pattern
  clAudioSourceController();

  //VVD fix lint
  ~clAudioSourceController();

   tBool WaitForSourceNeeded(clAudioSource &src);
   //tVoid trace(tChar *format, ...);
   static tVoid vTrace(tU16 u16TraceId,tU32 u32TraceData=0);
//   tBool bCleanUpStack();
//  tBool DemuteMute(switchingType_t switchType, clAudioSource *source);
   //tVoid vMuteStack_done();
   //tVoid vDemuteStack();
   //tVoid vMuteStack(clAudioSource*, tBool bForceStop=FALSE);
   //tBool bIsMuteStackFinished();
   tVoid vCrossOver();
   //clAudioSource* m_pclCurrentMuteIndex;

   //tU32 m_userData;
   //tU8 m_u8BgSourceInExchangeProcess;
   //tU8 m_u8State;

   //callback interface for audio-sources
   virtual tVoid vOn_done(clAudioSource*);
   virtual tVoid vOff_done(clAudioSource*);
   virtual tVoid vPause_done(clAudioSource*);
   virtual tVoid vInit_started(clAudioSource*);

   //tBool bIsAudiosourceAudible(clAudioSource* stackSource);

   clAudioSource* pcoGetTopOfStackAudiosource();
   clAudioSource* pcoGetTopOfStackAudiosource(clStack* pStack);
   clAudioSource* pcoGetTopOfStackEntertainmentAudiosource(clStack* pStack);

   clAudioSource* pcoGetCurrentMixSource();

public:
   clStack* getStackForSink(tU16 sinkID); // had to make it public to allow access from static methods

   tVoid vAddSinkClass(am_Sink_s sinkClass);
   tVoid vRemoveSinkClass(tU16 sinkClassID);


private:
   /** holds a pointer to the stack object */
   static clStack* m_pcoAudioStack;

   /** holds a pointer as a backup */
   static clStack* m_pcoBackupStack;

   // we introduce multiple stack, one for each sink
   static std::vector<clStack*> m_allAudioStacks;

   /** pointer to registered observer */
   static clIAudioSourceControllerObserver* m_poObserver;
   /** singleton pattern pointer */
   static clAudioSourceController* m_poSelf_Singleton;

   //create backup stack here
   am_Error_e vCreateBackupStack();
};

}}

#endif // _AUDIOSOURCECONTROLLER_H_
